--[[
Custom Speed Mods v3 (for StepMania 5)

changelog:

v3 (StepMania 5 b3)
* Complete rewrite to use profile load/save hooks.

--------------------------------------------------------------------------------
v2.3 (StepMania 5 a2/SM5TE) [by AJ]
* If someone has decided to remove 1x from the machine profile's speed mods,
  silently fix it.
* Ignore Cmod and mmod capitalization errors.

v2.2 (StepMania 5 alpha 2) [by FSX]
* Rewrite table management code.
* Add code to make sure that there are speed mods and that they are correct.

v2.1 (StepMania 5 Preview 2)
* Added support for m-Mods.

v2.0 (for sm-ssc)
Giant rewrite of the speed mod parser.
This rewrite comes with the following changes/features:
* Speed mods are now tied to profiles.
  This is arguably the biggest change, as it allows the speed mods to be
  portable, as well as per-profile.
  Thanks to this, we can now support reading SpeedMods from a USB stick or
  other external storage. (I didn't test writing yet, but it should work.)

This version of Custom Speed Mods will only run on StepMania 5 (due to m-mods).
--------------------------------------------------------------------------------
v1.4
* Try to auto-set the speed mod to 1.0 if:
 1) The player hasn't already chosen a speed mod
 2) The player's custom speed mod collection starts with a value under 1x.
 Due to the way the custom speed mods were coded, it will always pick the
 first value, even if it's not 1.0x.

v1.3
* strip whitespace out of file in case people use it.
	(I don't think it really works but SM seems to think the mods are legal)
* fixed an error related to using the fallback return value.

v1.2
* small fixes
* more comments

v1.1
* Cleaned up code some, I think.
]]
local ProfileSpeedMods = {}

-- Returns a new, empty mod table: a table with three members x, C, and m,
-- each being a table with the corresponding numbers set to true.
local function EmptyModTable()
	return {x = {}, C = {}, m = {}}
end

-- Merge one mod table into another.
local function MergeInModTable(dst, src)
	for typ, subtbl in pairs(src) do
		for n, v in pairs(subtbl) do
			dst[typ][n] = v
		end
	end
end

-- Parses a speed mod and returns the pair (type, number) or nil if parsing
-- failed.
local function CanonicalizeMod(mod)
	num = tonumber(mod:match("^(%d+.?%d*)[xX]$"))
	if num ~= nil then
		return "x", num
	end

	num = tonumber(mod:match("^[cC](%d+.?%d*)$"))
	if num ~= nil then
		return "C", num
	end

	num = tonumber(mod:match("^[mM](%d+.?%d*)$"))
	if num ~= nil then
		return "m", num
	end

	return nil
end

-- Parse a comma-separated string into a mod table.
local function StringToModTable(str)
	local mods = EmptyModTable()
	local valid = false

	string.gsub(str, "%s", "")
	for _, mod in ipairs(split(",", str)) do
		local t, n = CanonicalizeMod(mod)
		if t then
			mods[t][n] = true
			valid = true
		end
	end

	return valid and mods or nil
end

-- Return the contents of a mod table as a list of mod names.
local function ModTableToList(mods)
	local l = {}
	local tmp = {}

	-- Do x-mods separately because the x comes after
	for mod, _ in pairs(mods.x) do
		table.insert(tmp, mod)
	end
	table.sort(tmp)
	for _, mod in ipairs(tmp) do
		table.insert(l, mod .. "x")
	end

	-- C- and m-mods
	for _, modtype in ipairs({"C", "m"}) do
		tmp = {}
		for mod, _ in pairs(mods[modtype]) do
			table.insert(tmp, mod)
		end
		table.sort(tmp)
		for _, mod in ipairs(tmp) do
			table.insert(l, modtype .. mod)
		end
	end

	return l
end

local DefaultMods = StringToModTable("0.5x,1x,1.5x,2x,3x,4x,5x,6x,7x,8x,C250,C450,m550")

-- Reads the custom speed mod file at <path> and returns a corresponding mod
-- table.
local function ReadSpeedModFile(path)
	local file = RageFileUtil.CreateRageFile()
	if not file:Open(path, 1) then
		file:destroy()
		return nil
	end

	local contents = file:Read()
	file:Close()
	file:destroy()

	return StringToModTable(contents)
end

-- Hook called during profile load
function LoadProfileCustom(profile, dir)
	-- This will be (intentionally) nil if the file is missing or bad
	local mods = ReadSpeedModFile(dir .. "SpeedMods.txt")

	-- Special case for the machine profile
	if profile == PROFILEMAN:GetMachineProfile() then
		ProfileSpeedMods.machine = mods
		return
	end

	-- Otherwise, it's a player profile.  Store accordingly.
	for i = 1, NUM_PLAYERS do
		if profile == PROFILEMAN:GetProfile(PlayerNumber[i]) then
			ProfileSpeedMods[PlayerNumber[i]] = mods
			break
		end
	end
end

-- Hook called during profile save
function SaveProfileCustom(profile, dir)
	-- Change this if a theme allows you to change and save custom
	-- per-profile settings.
end

-- Returns a list of speed mods for the current round.
local function GetSpeedMods()
	-- Start with machine profile
	local mods = ProfileSpeedMods.machine or EmptyModTable()

	-- Merge in any active players
	for _, p in ipairs(GAMESTATE:GetHumanPlayers()) do
		if ProfileSpeedMods[p] and PROFILEMAN:IsPersistentProfile(p) then
			MergeInModTable(mods, ProfileSpeedMods[p])
		else
			MergeInModTable(mods, DefaultMods)
		end
	end

	-- Apparently removing 1x caused crashes, so be sure it's there.
	-- (This may not be a problem anymore. -- djpohly)
	mods.x[1] = true
	return ModTableToList(mods)
end

-- Implementation of custom Lua option row
function SpeedMods()
	local t = {
		Name = "Speed",
		LayoutType = "ShowAllInRow",
		SelectType = "SelectOne",
		OneChoiceForAllPlayers = false,
		ExportOnChange = true,
		Choices = GetSpeedMods(),

		LoadSelections = function(self, list, pn)
			local pref = GAMESTATE:GetPlayerState(pn):GetPlayerOptionsString("ModsLevel_Preferred")
			local selected = 0

			for i, choice in ipairs(self.Choices) do
				if string.find(pref, choice) then
					-- Found it, use it
					selected = i
					break
				elseif choice == "1x" then
					-- Pick this unless we find the
					-- preferred choice
					selected = i
				end
			end

			-- If we didn't find a match, just use the first
			if selected ~= 0 then
				list[selected] = true
			else
				list[1] = true
			end
		end,
		SaveSelections = function(self, list, pn)
			local state = GAMESTATE:GetPlayerState(pn)
			for i, choice in ipairs(self.Choices) do
				if list[i] then
					state:SetPlayerOptions("ModsLevel_Preferred", choice)
					return
				end
			end
			-- Or use the first
			state:SetPlayerOptions("ModsLevel_Preferred", self.Choices[1])
		end
	}
	return t
end

local default_speed_increment= 25
local default_speed_inc_large= 100

local function get_speed_increment()
	local increment= default_speed_increment
	if ReadGamePrefFromFile("SpeedIncrement") then
		increment= tonumber(GetGamePref("SpeedIncrement")) or default_speed_increment
	else
		WriteGamePrefToFile("SpeedIncrement", increment)
	end
	return increment
end

local function get_speed_inc_large()
	local inc_large= default_speed_inc_large
	if ReadGamePrefFromFile("SpeedIncLarge") then
		inc_large= tonumber(GetGamePref("SpeedIncLarge")) or default_speed_inc_large
	else
		WriteGamePrefToFile("SpeedIncLarge", inc_large)
	end
	return inc_large
end

function SpeedModIncSize()
	-- An option row for controlling the size of the increment used by
	-- ArbitrarySpeedMods.
	local increment= get_speed_increment()
	local ret= {
		Name= "Speed Increment",
		LayoutType= "ShowAllInRow",
		SelectType= "SelectMultiple",
		OneChoiceForAllPlayers= true,
		ExportOnChange = true,
		LoadSelections= function(self, list, pn)
			-- The first value is the status element, only it should be true.
			list[1]= true
		end,
		SaveSelections= function(self, list, pn)
			WriteGamePrefToFile("SpeedIncrement", increment)
		end,
		NotifyOfSelection= function(self, pn, choice)
			-- return true even though we didn't actually change anything so that
			-- the underlines will stay correct.
			if choice == 1 then return true end
			local incs= {10, 1, -1, -10}
			local new_val= increment + incs[choice-1]
			if new_val > 0 then
				increment= new_val
			end
			self:GenChoices()
			return true
		end,
		GenChoices= function(self)
			self.Choices= {tostring(increment), "+10", "+1", "-1", "-10"}
		end
	}
	ret:GenChoices()
	return ret
end

function SpeedModIncLarge()
	-- An option row for controlling the size of the increment used by
	-- ArbitrarySpeedMods.
	local inc_large= get_speed_inc_large()
	local ret= {
		Name= "Speed Increment Large",
		LayoutType= "ShowAllInRow",
		SelectType= "SelectMultiple",
		OneChoiceForAllPlayers= true,
		ExportOnChange = true,
		LoadSelections= function(self, list, pn)
			-- The first value is the status element, only it should be true.
			list[1]= true
		end,
		SaveSelections= function(self, list, pn)
			WriteGamePrefToFile("SpeedIncLarge", inc_large)
		end,
		NotifyOfSelection= function(self, pn, choice)
			-- return true even though we didn't actually change anything so that
			-- the underlines will stay correct.
			if choice == 1 then return true end
			local incs= {10, 1, -1, -10}
			local new_val= inc_large + incs[choice-1]
			if new_val > 0 then
				inc_large= new_val
			end
			self:GenChoices()
			return true
		end,
		GenChoices= function(self)
			self.Choices= {tostring(inc_large), "+10", "+1", "-1", "-10"}
		end
	}
	ret:GenChoices()
	return ret
end

function GetSpeedModeAndValueFromPoptions(pn)
	local poptions= GAMESTATE:GetPlayerState(pn):GetPlayerOptions("ModsLevel_Preferred")
	local speed= nil
	local mode= nil
	if poptions:MaxScrollBPM() > 0 then
		mode= "m"
		speed= math.round(poptions:MaxScrollBPM())
	elseif poptions:TimeSpacing() > 0 then
		mode= "C"
		speed= math.round(poptions:ScrollBPM())
	else
		mode= "x"
		speed= math.round(poptions:ScrollSpeed() * 100)
	end
	return speed, mode
end

function ArbitrarySpeedMods()
	-- If players are allowed to join while this option row is active, problems will probably occur.
	local increment= get_speed_increment()
	local inc_large= get_speed_inc_large()
	local ret= {
		Name= "Speed",
		LayoutType= "ShowAllInRow",
		SelectType= "SelectMultiple",
		OneChoiceForAllPlayers= false,
		ExportOnChange = false,
		LoadSelections= function(self, list, pn)
			-- The first values display the current status of the speed mod.
			if pn == PLAYER_1 or self.NumPlayers == 1 then
				list[1]= true
			else
				list[2]= true
			end
		end,
		SaveSelections= function(self, list, pn)
			local val= self.CurValues[pn]
			local poptions= GAMESTATE:GetPlayerState(pn):GetPlayerOptions("ModsLevel_Preferred")
			-- modify stage, song and current too so this will work in edit mode.
			local stoptions= GAMESTATE:GetPlayerState(pn):GetPlayerOptions("ModsLevel_Stage")
			local soptions= GAMESTATE:GetPlayerState(pn):GetPlayerOptions("ModsLevel_Song")
			local coptions= GAMESTATE:GetPlayerState(pn):GetPlayerOptions("ModsLevel_Current")
			if val.mode == "x" then
				local speed= val.speed / 100
				poptions:XMod(speed)
				stoptions:XMod(speed)
				soptions:XMod(speed)
				coptions:XMod(speed)
			elseif val.mode == "C" then
				poptions:CMod(val.speed)
				stoptions:CMod(val.speed)
				soptions:CMod(val.speed)
				coptions:CMod(val.speed)
			else
				poptions:MMod(val.speed)
				stoptions:MMod(val.speed)
				soptions:MMod(val.speed)
				coptions:MMod(val.speed)
			end
            MESSAGEMAN:Broadcast("ArbitrarySpeedModsSaved",{Player=pn})
		end,
		NotifyOfSelection= function(self, pn, choice)
			-- Adjust for the status elementsgit 
			local real_choice= choice - self.NumPlayers
			-- return true even though we didn't actually change anything so that
			-- the underlines will stay correct.
			if real_choice < 1 then return true end
			local val= self.CurValues[pn]
			if real_choice < 5 then
				local incs= {inc_large, increment, -increment, -inc_large}
				local new_val= val.speed + incs[real_choice]
				if new_val > 0 then
					val.speed= math.round(new_val)
				end
			elseif real_choice >= 5 then
				val.mode= ({"x", "C", "m"})[real_choice - 4]
			end
			self:GenChoices()
			MESSAGEMAN:Broadcast("SpeedChoiceChanged", {pn= pn, mode= val.mode, speed= val.speed})
			return true
		end,
		GenChoices= function(self)
			-- We can't show different options to each player, so compromise by
			-- only showing the xmod increments if one player is in that mode.
			local show_x_incs= false
			for pn, val in pairs(self.CurValues) do
				if val.mode == "x" then
					show_x_incs= true
				end
			end
			local big_inc= inc_large
			local small_inc= increment
			if show_x_incs then
				big_inc= tostring(big_inc / 100)
				small_inc= tostring(small_inc / 100)
			else
				big_inc= tostring(big_inc)
				small_inc= tostring(small_inc)
			end
			self.Choices= {
				"+" .. big_inc, "+" .. small_inc, "-" .. small_inc, "-" .. big_inc,
				"Xmod", "Cmod", "Mmod"}
			-- Insert the status element for P2 first so it will be second
			for i,pn in ipairs({PLAYER_2, PLAYER_1}) do
				local val= self.CurValues[pn]
				if val then
					if val.mode == "x" then
						table.insert(self.Choices, 1, (val.speed/100) .. "x")
					else
						table.insert(self.Choices, 1, val.mode .. val.speed)
					end
				end
			end
		end,
		CurValues= {}, -- for easy tracking of what speed the player wants
		NumPlayers= 0 -- for ease when adjusting for the status elements.
	}
	for i, pn in ipairs(GAMESTATE:GetEnabledPlayers()) do
		if GAMESTATE:IsHumanPlayer(pn) then
			local speed, mode= GetSpeedModeAndValueFromPoptions(pn)
			ret.CurValues[pn]= {mode= mode, speed= speed}
			ret.NumPlayers= ret.NumPlayers + 1
		end
	end
	ret:GenChoices()
	return ret
end

--[[
CustomSpeedMods (c) 2013 StepMania team.

Use freely, so long as this notice and the above documentation remains.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

Previous version was copyright © 2008-2012 AJ Kelly/KKI Labs.
]]



function ArbitrarySpeedMods2Increment()
	local increment = get_speed_increment()
	local inc_large = get_speed_inc_large()
    
    local function change_speed_mod_by_amount(pn, amount)
        local ps = GAMESTATE:GetPlayerState(pn)
        if not ps then
            lua.ReportScriptError("change_speed_mod_by_amount: No playerstate for "..pn..", ignoring request.")
            return
        end
        local preferred = ps:GetPlayerOptions("ModsLevel_Preferred")
        local stage = ps:GetPlayerOptions("ModsLevel_Stage")
        local song = ps:GetPlayerOptions("ModsLevel_Song")
        local current = ps:GetPlayerOptions("ModsLevel_Current")
        
        local func_name, value
        local value_needs_scaling_down = false
        --if this is 5.1, AMods won't be available, so don't check for them unless the PlayerOptions
        --supports them.
        if preferred.AMod and preferred:AMod() then
            value = preferred:AMod()
            func_name = 'AMod'
        elseif preferred:MMod() then
            value = preferred:MMod()
            func_name = 'MMod'
        elseif preferred:CMod() then
            value = preferred:CMod()
            func_name = 'CMod'
        elseif preferred:XMod() then
            value = preferred:XMod() * 100
            value_needs_scaling_down = true
            func_name = 'XMod'
        else
            lua.ReportScriptError("change_speed_mod_by_amount(): No recognized speed mod type set, can't modify speed mods.")
            return
        end
        
        value = value + amount
        --silently ignore attempts to reduce the set mod to a value at or below 0.
        if value <= 0 then
            return
        end
        if value_needs_scaling_down then
            value = value / 100
        end
        
        preferred[func_name](preferred, value)
        stage[func_name](stage, value)
        song[func_name](song, value)
        current[func_name](current, value)
        MESSAGEMAN:Broadcast("SpeedModChanged",{PlayerNumber=pn})
    end
    local values = {inc_large, increment, -increment, -inc_large}
    return {
        Name = "SpeedIncrement",
        LayoutType = "ShowAllInRow",
        SelectType = "SelectMultiple",
        OneChoiceForAllPlayers = false,
        ExportOnChange = true,
        Choices = {tostring(values[1]), tostring(values[2]), tostring(values[3]), tostring(values[4])},
        LoadSelections = function(self, choice_list, pn)
            for idx, _ in pairs(choice_list) do
                choice_list[idx] = false
            end
        end,
        --dummy function so SM doesn't crash.
        SaveSelections = function() end,
        NotifyOfSelection = function(self, pn, choice)
            change_speed_mod_by_amount(pn, values[choice])
            --this hack causes SM to call LoadSelections, which clears the choice list.
            return true
        end
    }
        
        
end

function ArbitrarySpeedMods2ModType()
    --In this case, we need to change the choices list depending on the presence of AMods
    local has_AMod = PlayerOptions.AMod ~= nil
    local choices
    if has_AMod then
        choices = {'XMod', 'CMod', 'MMod', 'AMod'}
    else
        choices = {'XMod', 'CMod', 'MMod'}
    end
    
    local function change_speed_mod_type(pn, func_name)
        local ps = GAMESTATE:GetPlayerState(pn)
        if not ps then
            lua.ReportScriptError("change_speed_mod_type: No playerstate for "..pn..", ignoring request.")
            return
        end
        local preferred = ps:GetPlayerOptions("ModsLevel_Preferred")
        if not preferred[func_name] then
            lua.ReportScriptError("change_speed_mod_by_amount(): invalid option function name "..tostring(func_name))
            return
        end
        local stage = ps:GetPlayerOptions("ModsLevel_Stage")
        local song = ps:GetPlayerOptions("ModsLevel_Song")
        local current = ps:GetPlayerOptions("ModsLevel_Current")
        local value = 100
        if has_AMod and preferred:AMod() then
            value = preferred:AMod()
        elseif preferred:MMod() then
            value = preferred:MMod()
        elseif preferred:CMod() then
            value = preferred:CMod()
        elseif preferred:XMod() then
            value = preferred:XMod() * 100
        else
            lua.ReportScriptError("change_speed_mod_by_amount(): No recognized speed mod type set, can't modify speed mods.")
            return
        end
        if func_name == "XMod" then
            value = value / 100
        end
        --To clarify, this does the same thing as [po obj]:[insert func_name here](value)
        preferred[func_name](preferred, value)
        stage[func_name](stage, value)
        song[func_name](song, value)
        current[func_name](current, value)
        MESSAGEMAN:Broadcast("SpeedModChanged",{PlayerNumber=pn})
    end
    
    return {
        Name = "SpeedType",
        LayoutType = "ShowAllInRow",
        SelectType = "SelectMultiple",
        OneChoiceForAllPlayers = false,
        ExportOnChange = true,
        Choices = choices,
        LoadSelections = function(self, choice_list, pn)
            for idx, _ in pairs(choice_list) do
                choice_list[idx] = false
            end
        end,
        --dummy function so SM doesn't crash.
        SaveSelections = function() end,
        NotifyOfSelection = function(self, pn, choice)
            change_speed_mod_type(pn, choices[choice])
            --as above
            return true
        end
    }
end

function OptionRowScreenFilter()
	--we use integers equivalent to the alpha value multiplied by 10
	--to work around float precision issues
	local choiceToAlpha = {0, 20, 40, 60, 80, 100}
	local alphaToChoice = {[0]=1, [20]=2, [40]=3, [60]=4, [80]=5, [100]=6}
	local ProfilePrefs = LoadModule"ProfilePrefs.lua"
	local t = {
		Name="Filter",
		LayoutType = "ShowAllInRow",
		SelectType = "SelectOne",
		OneChoiceForAllPlayers = false,
		ExportOnChange = true,
		Choices = { "0%", "20%", "40%", "60%", "80%", "100%"},
		LoadSelections = function(self, list, pn)
			local pName = ToEnumShortString(pn)
			local profileID = GetProfileIDForPlayer(pn)
			local pPrefs = ProfilePrefs.Read(profileID)
			local filterValue = pPrefs.filter
			if filterValue ~= nil then
				local val = alphaToChoice[filterValue] or 1
				list[val] = true
			else
				list[1] = true
			end
		end,
		SaveSelections = function(self, list, pn)
			local pName = ToEnumShortString(pn)
			local found = false
			for i=1,#list do
				if not found then
					if list[i] == true then
						local profileID = GetProfileIDForPlayer(pn)
						local pPrefs = ProfilePrefs.Read(profileID)
						pPrefs.filter = choiceToAlpha[i]
						ProfilePrefs.Save(profileID)
						found = true
					end
				end
			end
		end,
	};
	setmetatable(t, t)
	return t
end

local GetModsAndPlayerOptions = function(player)
	local mods = GAMESTATE:GetPlayerState(pn):GetPlayerOptions('ModsLevel_Preferred')
	local topscreen = SCREENMAN:GetTopScreen():GetName()
	local modslevel = topscreen  == "ScreenEditOptions" and "ModsLevel_Stage" or "ModsLevel_Preferred"
	local playeroptions = GAMESTATE:GetPlayerState(player):GetPlayerOptions(modslevel)

	return mods, playeroptions
end

function OptionRowGuideLine()
	local ProfilePrefs = LoadModule"ProfilePrefs.lua"
	local t = {
		Name="GuideLine",
		LayoutType = "ShowAllInRow",
		SelectType="SelectOne",
		OneChoiceForAllPlayers=false,
		ExportOnChange=true,
		Choices = {"On","Off"},
		Values = {true,false},
		LoadSelections = function(self,list,pn)
			local profileID = GetProfileIDForPlayer(pn)
			local pPrefs = ProfilePrefs.Read(profileID)
			if pPrefs.guidelines == true then
				list[1] = true
			elseif pPrefs.guidelines == false then
				list[2] = true
			else
				list[1] = true
			end
		end,
		SaveSelections = function(self,list,pn)
			for i, value in ipairs(self.Values) do
				if list[i] then
					local profileID = GetProfileIDForPlayer(pn)
					local pPrefs = ProfilePrefs.Read(profileID)
					pPrefs.guidelines = value
					ProfilePrefs.Save(profileID)
				end
			end
		end,	
	};
	setmetatable(t ,t)
	return t
end

function OptionRowBias()
	local ProfilePrefs = LoadModule"ProfilePrefs.lua"
	local t = {
		Name="Bias",
		LayoutType = "ShowAllInRow",
		SelectType="SelectOne",
		OneChoiceForAllPlayers=false,
		ExportOnChange=true,
		Choices = {"On","Off"},
		Values = {true,false},
		LoadSelections = function(self,list,pn)
			local profileID = GetProfileIDForPlayer(pn)
			local pPrefs = ProfilePrefs.Read(profileID)
			if pPrefs.bias == true then
				list[1] = true
			elseif pPrefs.bias == false then
				list[2] = true
			else
				list[1] = true
			end
		end,
		SaveSelections = function(self,list,pn)
			for i, value in ipairs(self.Values) do
				if list[i] then
					local profileID = GetProfileIDForPlayer(pn)
					local pPrefs = ProfilePrefs.Read(profileID)
					pPrefs.bias = value
					ProfilePrefs.Save(profileID)
				end
			end
		end,	
	};
	setmetatable(t ,t)
	return t
end

function OptionRowEX()
	local ProfilePrefs = LoadModule"ProfilePrefs.lua"
	local t = {
		Name="EX",
		LayoutType = "ShowAllInRow",
		SelectType="SelectOne",
		OneChoiceForAllPlayers=false,
		ExportOnChange=true,
		Default = false,
		Choices = {"Money Score","EXSCORE"},
		Values = {false,true},
		LoadSelections = function(self,list,pn)
			local profileID = GetProfileIDForPlayer(pn)
			local pPrefs = ProfilePrefs.Read(profileID)
			if pPrefs.ex_score == false then
				list[1] = true
			elseif pPrefs.ex_score == true then
				list[2] = true
			else
				list[1] = true
			end
		end,
		SaveSelections = function(self,list,pn)
			for i, value in ipairs(self.Values) do
				if list[i] then
					local profileID = GetProfileIDForPlayer(pn)
					local pPrefs = ProfilePrefs.Read(profileID)
					pPrefs.ex_score = value
					ProfilePrefs.Save(profileID)
				end
			end
		end,	
	};
	setmetatable(t ,t)
	return t
end

function OptionRowScoreLab()
	local ProfilePrefs = LoadModule"ProfilePrefs.lua"
	local t = {
		Name="ScoreLabel",
		LayoutType = "ShowAllInRow",
		SelectType="SelectOne",
		OneChoiceForAllPlayers=false,
		ExportOnChange=true,
		Default = false,
		Choices = {"Profile","BPM"},
		Values = {"Profile","BPM"},
		LoadSelections = function(self,list,pn)
			local profileID = GetProfileIDForPlayer(pn)
			local pPrefs = ProfilePrefs.Read(profileID)
			if pPrefs.scorelabel == "Profile" then
				list[1] = true
			elseif pPrefs.scorelabel == "BPM" then
				list[2] = true
			else
				list[1] = true
			end
		end,
		SaveSelections = function(self,list,pn)
			for i, value in ipairs(self.Values) do
				if list[i] then
					local profileID = GetProfileIDForPlayer(pn)
					local pPrefs = ProfilePrefs.Read(profileID)
					pPrefs.scorelabel = value
					ProfilePrefs.Save(profileID)
				end
			end
		end,	
	};
	setmetatable(t ,t)
	return t
end

function stringify( tbl, form )
	if not tbl then return end

	local t = {}
	for _,value in ipairs(tbl) do
		t[#t+1] = (type(value)=="number" and form and form:format(value) ) or tostring(value)
	end
	return t
end

function MiniSelector()
	local t = {
		Name="Mini",
		LayoutType = "ShowOneInRow",
		SelectType = "SelectOne",
		OneChoiceForAllPlayers = false,
		ExportOnChange = true,
		Choices = stringify(fornumrange(-100,100,5), "%g%%"),
		LoadSelections=function(self,list,pn)
			setenv("NumMini",#list)
			local nearest_i
			local best_difference = math.huge
			for i,v2 in ipairs(self.Choices) do
				local mini = GAMESTATE:GetPlayerState(pn):GetPlayerOptions("ModsLevel_Preferred"):Mini()
				local this_diff = math.abs(mini - v2:gsub("(%d+)%%", tonumber) / 100)
				if this_diff < best_difference then
					best_difference = this_diff
					nearest_i = i
				end
			end
			list[nearest_i] = true
		end,
		SaveSelections=function(self,list,pn)
			for i, choice in ipairs(self.Choices) do
				if list[i] then
					local poptions= GAMESTATE:GetPlayerState(pn):GetPlayerOptions("ModsLevel_Preferred")
					local stoptions= GAMESTATE:GetPlayerState(pn):GetPlayerOptions("ModsLevel_Stage")
					local soptions= GAMESTATE:GetPlayerState(pn):GetPlayerOptions("ModsLevel_Song")
					local coptions= GAMESTATE:GetPlayerState(pn):GetPlayerOptions("ModsLevel_Current")
					local mini = string.gsub(choice,"%%","")/100
					poptions:Mini(mini)
					stoptions:Mini(mini)
					soptions:Mini(mini)
					coptions:Mini(mini)
				end
			end
		end,
	}
	return t
end

function MusicRate()
	local increment = 0.025
	local inc_large= 0.1
	local t = {
		Name="Rate",
		LayoutType="ShowAllInRow",
		SelectType="SelectOne",
		Choices = stringify(fornumrange(10,200,5), "%g%%"),
		OneChoiceForAllPlayers=true,
		ExportOnChange=true,
		LoadSelections=function(self,list, pn)
			local nearest_i
			local best_difference = math.huge
			setenv("NumRate",#list)
			for i,v2 in ipairs(self.Choices) do
				local rate = GAMESTATE:GetSongOptionsObject("ModsLevel_Preferred"):MusicRate()
				local this_diff = math.abs(rate - v2:gsub("(%d+)%%", tonumber) / 100)
				if this_diff < best_difference then
					best_difference = this_diff
					nearest_i = i
				end
			end
			list[nearest_i] = true
		end,
		SaveSelections = function(self,list,pn)
			for i,choice in ipairs(self.Choices) do
				if list[i] then
					local MR = string.gsub(self.Choices[i],"%%","")/100
					GAMESTATE:GetSongOptionsObject("ModsLevel_Preferred"):MusicRate(MR)
				end
			end
		end,
	};
	return t
end
